//
//  AHKPhone.m
//  AH-K3001V Utilities
//
//  Created by FUJIDANA on 06/04/21.
//  Copyright 2006 FUJIDANA. All rights reserved.
//
//
//  AirHPhone.m
//  BookmarkUtility
//
//  Created by raktajino on Thu Jun 17 2004.
//  Copyright (c) 2004 raktajino. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

#import "AHKPhone.h"

#import <stdio.h>
#import <unistd.h>
#import <errno.h>
#import <fcntl.h>
#import <sys/ioctl.h>

#define kOpeningRetryCount			8						// オープン／ログイン時の最大リトライ回数
#define kOpeningRetryInterval		0.25					// オープン／ログイン時のリトライ間隔（単位：秒）
#define kRequireDriverMajorVersion	1						// AH-K3001V USB Driverのメジャー・バージョン番号の要求値
#define kRequireDriverMinorVersion	2						// AH-K3001V USB Driverのマイナー・バージョン番号の要求値

#define kMaxPersonCount			500
#define kMaxMailCount			1000

NSString *AHKErrorDomain = @"jp.ne.dion.k2.fujidana.AHKErrorDomain";

@interface AHKPhone (Private)

- (SInt32)sendRawData:(UInt8 *)buffer length:(SInt32)length error:(NSError **)errorPtr;

- (BOOL)logoutError:(NSError **)errorPtr;

@end

@implementation AHKPhone

// --- check whtether the driver has been installed. Return yes if the driver exists.

+ (BOOL)existsDriver
{
	NSArray *paths = NSSearchPathForDirectoriesInDomains(NSLibraryDirectory, NSSystemDomainMask, YES);
	if (paths == nil || [paths count] == 0) return NO;
	
	NSString *driverPath = [[[paths lastObject] stringByAppendingPathComponent:@"Extensions"] stringByAppendingPathComponent:@"AHK3001VDriver.kext"];
	NSBundle *driverBundle = [NSBundle bundleWithPath:driverPath];
	if(driverBundle == nil) return NO;
	
	NSString *versionString = [driverBundle objectForInfoDictionaryKey: @"CFBundleShortVersionString"];
	if(versionString == nil) return NO;
	
	NSArray *versionArray = [versionString componentsSeparatedByString:@"."];
	if([versionArray count] < 2) return NO;
	
	SInt32 currentMajorVersion = [[versionArray objectAtIndex:0] intValue];
	SInt32 currentMinorVersion = [[versionArray objectAtIndex:1] intValue];
	
	if(currentMajorVersion > kRequireDriverMajorVersion) {
		return YES;
	}
	else if(currentMajorVersion == kRequireDriverMajorVersion && currentMinorVersion >= kRequireDriverMinorVersion) {
		return YES;
	}
	return NO;
}

// --- get name of required driver.

+ (NSString *)humanReadableDriverName
{
	return [NSString stringWithFormat:@"AH-K3001V USB Driver %d.%d", kRequireDriverMajorVersion, kRequireDriverMinorVersion];
}

// --- initialize object

- (id)init
{
	self = [super init];
	if (self != nil) {
		fileDescriptor = -1;
		loggedin       = NO;
	}
	return self;
}

// --- deallocate object.

- (void)dealloc
{
	[self closeError:NULL];
	
	[super dealloc];
}

// --- open connection to cellphone.

- (BOOL)openError:(NSError **)errorPtr
{
	// open serial port
	int n = 0;
	int errorNum = 0;
	while (true) {
		fileDescriptor = open("/dev/cu.ah-k3001v.data", O_RDWR | O_NOCTTY | O_NONBLOCK);
		if (fileDescriptor == -1) {
			errorNum = errno;
			if (errorNum == ENOENT) {
				if (n++ < kOpeningRetryCount) {
					[NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:kOpeningRetryInterval]]; 
					continue;
				}
				//@"The AIR-EDGE PHONE wasn't connected."
				if (errorPtr != NULL) {
					NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"The AIR-EDGE PHONE wasn't connected.", @"error.phoneConnection.description")
																		 forKey:NSLocalizedDescriptionKey];
					*errorPtr = [NSError errorWithDomain:NSPOSIXErrorDomain 
													code:errorNum
												userInfo:userInfo];
				}
				return NO;
			} else {
				//@"Couldn't open a port for the AIR-EDGE PHONE."
				if (errorPtr != NULL) {
					NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"Couldn't open a port for the AIR-EDGE PHONE.", @"error.openPort.description")
																		 forKey:NSLocalizedDescriptionKey];
					*errorPtr = [NSError errorWithDomain:NSPOSIXErrorDomain 
													code:errorNum
												userInfo:userInfo];
				}
				return NO;
			}
		}
		break;
	}
	
	// set the serial port for exclusive use
	if (ioctl(fileDescriptor, TIOCEXCL) == -1) {
		errorNum = errno;
		goto error;
	}
	
	if (fcntl(fileDescriptor, F_SETFL, 0) == -1) {
		errorNum = errno;
		goto error;
	}
	
	// get and store the current attribute of the serial port
	if (tcgetattr(fileDescriptor, &originalTTYAttrs) == -1) {
		errorNum = errno;
		goto error;
	}
	
	// change the attribute of the serial port
	struct termios options = originalTTYAttrs;
	options.c_cflag = CS8 | CREAD | HUPCL | CLOCAL | CRTSCTS;
	options.c_iflag = IGNBRK | IGNPAR;
	options.c_oflag = 0;
	options.c_lflag = 0;
	for (n = 0; n < NCCS; n++) {
		options.c_cc[n] = 0;
	}
	options.c_cc[VMIN]  = 1;
	options.c_cc[VTIME] = 0;
	if (tcsetattr(fileDescriptor, TCSANOW, &options) == -1) {
		errorNum = errno;
		goto error;
	}
	
	loggedin = NO;
	return YES;
	
error:
		//@"Couldn't set the serial port options."
		if (fileDescriptor != -1) {
			close(fileDescriptor);
			fileDescriptor = -1;
		}
	
	if (errorPtr != NULL) {
		NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"Couldn't set the serial port options.", @"error.setSerialPortOptions.description")
															 forKey:NSLocalizedDescriptionKey];
		*errorPtr = [NSError errorWithDomain:NSPOSIXErrorDomain 
										code:errorNum
									userInfo:userInfo];
	}
	return NO;
}

// --- close connection to cellphone

- (BOOL)closeError:(NSError **)errorPtr
{
	BOOL flag = YES;
	if (fileDescriptor != -1) {
		flag = [self logoutError:errorPtr];
		tcdrain(fileDescriptor);
		tcsetattr(fileDescriptor, TCSANOW, &originalTTYAttrs);
		close(fileDescriptor);
		fileDescriptor = -1;
	}
	return flag;
}

// --- send command to cellphone

- (BOOL)sendCommand:(SInt32)a subCommand:(SInt32)b parameter:(char *)parameter error:(NSError **)errorPtr
{
	SInt32 length = strlen(parameter);
	return [self sendCommand:a subCommand:b parameter:(UInt8 *)parameter length:length error:errorPtr];
}


- (BOOL)sendCommand:(SInt32)a subCommand:(SInt32)b parameter:(UInt8 *)parameter length:(SInt32)length error:(NSError **)errorPtr
{
	UInt8 buf[12] = {
		0xe1, 0x01, (UInt8)(a >> 8), (UInt8)a,
		0, 0, 0, 0,
		0, (UInt8)(b >> 8), 0, (UInt8)b
	};
	
	const SInt32 lenWithHeader = length + 4;
	buf[4] = (UInt8) lenWithHeader;
	buf[5] = (UInt8)(lenWithHeader >> 8 );
	buf[6] = (UInt8)(lenWithHeader >> 16);
	buf[7] = (UInt8)(lenWithHeader >> 24);
	
	if ([self sendRawData:buf length:sizeof(buf) error:errorPtr] == -1) return NO;
	
	if(parameter != nil) {
		if ([self sendRawData:parameter length:length error:errorPtr] == -1) return NO;
	}
	return YES;
}

// --- receive data from cellphone

- (SInt32)receiveData:(UInt8 *)buffer maxLength:(SInt32)bufferSize additionally:(BOOL)hasAdditionallyData error:(NSError **)errorPtr
{
	// write end mark on buffer
	buffer[12] = 0xff;
	buffer[13] = 0x00;
	
	int bufferLength = 0;
	while (bufferLength < 12) {
		// receive data
		SInt32 numBytes = [self receiveRawData:(buffer + bufferLength) maxLength:(12 - bufferLength) error:errorPtr];
		if (numBytes == -1) {
			return -1;
		}
		
		bufferLength += numBytes;
		
		// seek header mark
		int n;
		for (n = 0; n < bufferLength - 1; n++) {
			if (buffer[n] == 0xe1 && buffer[n + 1] == 0x02) {
				break;
			}
		}
		
		// if header mark is found, move buffer data so that header mark becomes the beginning of buffer
		if (n > 0) {
			bufferLength -= n;
			memmove(buffer, buffer + n, bufferLength);
		}
	}
	
	const int len = buffer[4] + (((SInt32)buffer[5]) << 8) + (((SInt32)buffer[6]) << 16) + (((SInt32)buffer[7]) << 24) + 8;
	
	if (hasAdditionallyData) {
		if(len < 12) {
			//			[NSException raise:kAirHPhoneException format:@"An illiegal length data was received."];
			NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"An illegal length data was received.", @"phone.InapplicableDataReceivedError.description")
																 forKey:NSLocalizedDescriptionKey];
			*errorPtr = [NSError errorWithDomain:AHKErrorDomain
											code:AHKPhoneInapplicableDataReceivedError
										userInfo:userInfo];
			return -1;
		}
		
		const int size = (len > bufferSize) ? bufferSize : len;
		bufferLength = 12;
		while (bufferLength < size) {
			SInt32 numBytes = [self receiveRawData:buffer + bufferLength maxLength:size - bufferLength error:errorPtr];
			if (numBytes == -1) {
				return -1;
			}
			bufferLength += numBytes;
		}
	}
	return len;
}

// --- AirH" Phoneへデータを無加工のまま送信する。

- (SInt32)sendRawData:(UInt8 *)buffer length:(SInt32)length error:(NSError **)errorPtr
{
	SInt32 numBytes = write(fileDescriptor, buffer, length);
	if (numBytes != length) {
		// @"Couldn't write the serial port."
		//		if (numBytes == -1) {
		if (errorPtr != NULL) {
			NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"Couldn't write the serial port.", @"error.sendBytes.description") 
																 forKey:NSLocalizedDescriptionKey];
			
			*errorPtr = [NSError errorWithDomain:NSPOSIXErrorDomain
											code:errno
										userInfo:userInfo];
		}
		return -1;
		//		}
	}
	return numBytes;
}

// --- AirH" Phoneからデータを無加工のまま受信する。

- (SInt32)receiveRawData:(UInt8 *)buffer maxLength:(SInt32)bufferSize error:(NSError **)errorPtr
{
	SInt32 numBytes = read(fileDescriptor, buffer, bufferSize);
	if(numBytes == -1) {
		//		[NSException raise:kAirHPhoneException format:@"Couldn't read the serial port."];
		if (errorPtr != NULL) {
			NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"Couldn't read the serial port.", @"error.receiveBytes.description") 
																 forKey:NSLocalizedDescriptionKey];
			*errorPtr = [NSError errorWithDomain:NSPOSIXErrorDomain
											code:errno
										userInfo:userInfo];
		}
		return -1;
	}
	return numBytes;
}

// --- AirH" Phoneへログインする。

- (BOOL)loginWithPassword:(NSString *)password error:(NSError **)errorPtr
{
	UInt8 buffer[kReadBufferSize];
	
	// ログインする。
	int n = 0;
	while (true) {
		if ([self sendCommand:0 subCommand:0x0101 parameter:"1" error:errorPtr] == NO) {
			return NO;
		}
		if ([self receiveData:buffer maxLength:sizeof(buffer) additionally:YES error:errorPtr] == -1) {
			return NO;
		}
		
		if (strncmp((char *)buffer + 12, "OK", 2)) {
			if (n++ < kOpeningRetryCount) {
				[NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:kOpeningRetryInterval]]; 
				continue;
			}
			if (errorPtr != NULL) {
				NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"The AIR-EDGE PHONE wasn't the top screen.", @"error.notTopScreen.description")
																	 forKey:NSLocalizedDescriptionKey];
				*errorPtr = [NSError errorWithDomain:AHKErrorDomain
												code:AHKPhoneNotReadyError
											userInfo:userInfo];
			}
			return NO;
			//			[NSException raise:kAirHPhoneException format:@"The AIR-EDGE PHONE wasn't the top screen."];
		}
		break;
	}
	loggedin = YES;
	
	// convert password to C-string
	NSData *passwordData = [password dataUsingEncoding:NSShiftJISStringEncoding];
	if(passwordData == nil) {
		// in case password contains characters which can be converted to SJIS encoding
		if (errorPtr != NULL) {
			NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"The password contains incorrect characters.", @"error.passswordStringEncoding.description")
																 forKey:NSLocalizedDescriptionKey];
			*errorPtr = [NSError errorWithDomain:AHKErrorDomain
											code:AHKPhoneInapplicablePasswordStringEncodingError
										userInfo:userInfo];
		}
		return NO;
	}
	int passwordDataLength = [passwordData length];
	if(passwordDataLength <= 0 || passwordDataLength > 4) {
		// in case password length is 0 or longer than 4
		if (errorPtr != NULL) {
			NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"The password must consist of just 4 characters.", @"error.passswordStringLength.description")
																 forKey:NSLocalizedDescriptionKey];
			*errorPtr = [NSError errorWithDomain:AHKErrorDomain
											code:AHKPhoneInapplicablePasswordStringLengthError
										userInfo:userInfo];
		}
		return NO;
	}
	[passwordData getBytes:buffer length:passwordDataLength];
	buffer[passwordDataLength] = 0;
	
	// send password to cellphone
	if ([self sendCommand:1 subCommand:0x0201 parameter:(char *)buffer error:errorPtr] == NO) {
		return NO;
	}
	if ([self receiveData:buffer maxLength:sizeof(buffer) additionally:YES error:errorPtr] == NO) {
		return NO;
	}
	if(strncmp((char *)buffer + 12, "OK", 2)) {
		if (errorPtr != NULL) {
			NSDictionary *userInfo = [NSDictionary dictionaryWithObject:NSLocalizedString(@"Password is incorrect.", @"error.incorrectPassword.description") 
																 forKey:NSLocalizedDescriptionKey];
			*errorPtr = [NSError errorWithDomain:AHKErrorDomain
											code:AHKPhoneIncorrectPasswordError
										userInfo:userInfo];
		}
		return NO;
	}
	return YES;
}

- (BOOL)logoutError:(NSError **)error
{
	if(loggedin) {
		if ([self sendCommand:0xff subCommand:0x0102 parameter:"" error:error] == NO) {
			return NO;
		}
		loggedin = NO;
		return YES;
	} else {
		if (error != NULL) {
			*error = [NSError errorWithDomain:AHKErrorDomain
										 code:AHKPhoneNotLoggedInError
									 userInfo:nil];
		}
		return NO;
	}
}

@end
